function stdcell_lwire()
% L-WIRE: L-shaped structure with TWO perpendicular branches
% Keeps only:
%  - Positive Y arm (3 cells)
%  - Rotated positive X arm (3 cells)
% Removes:
%  - Negative Y arm → replaced by a LARGE FILLER (covering SW + SE)
%  - Negative X arm → replaced by a LARGE FILLER (covering NW + NE)

%% ===================== PARAMETERS ===============================
P = struct( ...
    'L',24, 'W',10, 'H',2, ...                   % SiO2 base
    'origin',[0 0 0], ...                        % [x0 y0 z0]
    'H_Ti_bottom',1, ...                         % Ti bottom (only H)
    'H_gold_bottom',2, ...                       % Au bottom (only H)
    'L_vac',4, 'H_vac',3, ...                    % vacuum (Lx, Hz)
    'L_Au_top',16, 'W_Au_top',8, 'H_Au_top',4, ... % Au top (Lx, Wy, Hz) for standard cells
    'T_film',0.5, ...                            % Ti film thickness
    'CapSide_center',22 ...                      % << Au cap side of the CENTRAL BLOCK (nm)
);

% Colors / style
c_base=[0.55 0.35 0.20];    % SiO2 (opaque brown)
c_ti=[0.20 0.20 0.20];      % Ti (dark gray)
c_au=[1.00 0.84 0.00];      % Au (opaque yellow)
c_vac=[0.00 0.60 0.00];     % Vacuum (green)
a_vac=0.35;
c_hfo2=[0.80 0.65 0.45];    % HfO2 (light brown)
edgeCol='k'; lw=1.0;

F = faces6();

%% ============== PRE-COMPUTATIONS (pitch and heights) =================
pitch  = calcPitchCutY(P);  % width of CUT cell (after Y cut on far side)
H_tot  = P.H + P.H_Ti_bottom + P.H_gold_bottom + P.H_vac + P.H_Au_top;

% Central block: side = L (square L x L seen from above)
L_block = P.L;
W_block = P.L;

% Global origin
x0 = P.origin(1); y0 = P.origin(2); z0 = P.origin(3);

% === Origins of the "vertical" structure (Y axis) – ONLY POSITIVE ARM ===
o1 = [x0, y0,             z0];
o2 = [x0, y0 + pitch,     z0];
o3 = [x0, y0 + 2*pitch,   z0];
% block immediately after the STANDARD cell
oB = [x0, y0 + 2*pitch + P.W, z0];

% Center of the block (rotation axis in the x–y plane)
Crot = [oB(1) + L_block/2, oB(2) + W_block/2];

%% ============== GEOMETRY CONSTRUCTION (VERTICES) =================
% Cells of the positive Y arm
G1 = buildCellVerts(P, o1, true );
G2 = buildCellVerts(P, o2, true );
G3 = buildCellVerts(P, o3, false);

% >>> CENTRAL BLOCK: STANDARD SQUARE CELL <<<
Gc = buildCentralSquareCellVerts(P, oB);

% Rotated copy by +90° around Crot (Z axis) – positive X arm only
Rz = [0 -1 0; 1 0 0; 0 0 1];  % +90°
G1r = rotZ_around(G1, Rz, Crot);
G2r = rotZ_around(G2, Rz, Crot);
G3r = rotZ_around(G3, Rz, Crot);

%% ============== CORNER FILLERS (as in MV) =================
arm_span = P.W + 2*pitch;   % extension of an arm beyond the block edge

% Useful coordinates of the block
xB = oB(1);           yB = oB(2);
xB2 = xB + L_block;   yB2 = yB + W_block;

% Opaque layers with required heights:
h1 = P.H + P.H_Ti_bottom;   % layer 1  (SiO2 color)
h2 = P.H_gold_bottom;       % layer 2  (Au color)
h3 = P.H_vac;               % layer 3  (HfO2 color)
h4 = P.H_Au_top;            % layer 4  (SiO2 color)

% For L-wire: the arms meet in NW, so fillers in NE, SW, SE, and NW
o_NE = [xB2,            yB2,            z0];
o_SW = [xB - arm_span,  yB - arm_span,  z0];
o_SE = [xB2,            yB - arm_span,  z0];
o_NW = [xB - arm_span,  yB2,  z0];

FNE = buildFillerLayers(o_NE, arm_span, arm_span, [h1 h2 h3 h4]);
FSW = buildFillerLayers(o_SW, arm_span, arm_span, [h1 h2 h3 h4]);
FSE = buildFillerLayers(o_SE, arm_span, arm_span, [h1 h2 h3 h4]);
FNW = buildFillerLayers(o_NW, arm_span, arm_span, [h1 h2 h3 h4]);

% === STRIP for negative Y arm (horizontal below the block) ===
o_Y_strip = [xB, yB2, z0];
Lx_Y_strip = L_block;        % block width
Wy_Y_strip = arm_span;       % arm length
FY_strip = buildFillerLayers(o_Y_strip, Lx_Y_strip, Wy_Y_strip, [h1 h2 h3 h4]);

% === STRIP for negative X arm (vertical to the left of the block) ===
o_X_strip = [xB - arm_span, yB, z0];
Lx_X_strip = arm_span;       % arm length
Wy_X_strip = W_block;        % block height
FX_strip = buildFillerLayers(o_X_strip, Lx_X_strip, Wy_X_strip, [h1 h2 h3 h4]);

%% ======================== PLOT =================================
fig = figure('Color','w'); ax = axes(fig);
cla(ax,'reset'); hold(ax,'on');

% --- Draw positive Y arm structure ---
drawGeom(ax,G1, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, true );   % legend only once
drawGeom(ax,G2, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax,G3, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);

% --- Central block ---
drawGeom(ax,Gc, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);

% --- Draw rotated copy (positive X arm) ---
drawGeom(ax,G1r, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax,G2r, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);
drawGeom(ax,G3r, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);

% --- Layered fillers at the corners (NE, SW, SE, NW, and strips) ---
drawFillerLayers(ax, FNE, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, 'FILLER (layered)');
drawFillerLayers(ax, FSW, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
drawFillerLayers(ax, FSE, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
drawFillerLayers(ax, FNW, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
drawFillerLayers(ax, FY_strip, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
drawFillerLayers(ax, FX_strip, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);

% Axes & view
axis(ax,'equal'); grid(ax,'on'); view(ax,35,25);
xlabel(ax,'X [nm]'); ylabel(ax,'Y [nm]'); zlabel(ax,'Z [nm]');
title(ax, sprintf('Majority Voter Vacuum- L=%.3g nm – H_{tot}=%.3g nm – Cap_{center}=%.3g nm', ...
    P.L, H_tot, P.CapSide_center));

% Convenient limits (same dimensions as MV)
Y_span = (2*pitch + P.W) + W_block + (P.W + 2*pitch);
X_span = Y_span;
marg   = 0.12 * [X_span, Y_span, H_tot];
xlim(ax,[x0 - arm_span - marg(1), x0 + L_block + arm_span + marg(1)]);
ylim(ax,[y0 - arm_span - marg(2), y0 + Y_span + marg(2)]);
zlim(ax,[z0 - marg(3), z0 + H_tot + marg(3)]);

% Origin
plot3(ax,x0,y0,z0,'ko','MarkerFaceColor','k','HandleVisibility','off');

legend(ax,'Location','bestoutside'); hold(ax,'off');
end

% ======================================================================
% ===================== SUBFUNCTIONS ===================================
% ======================================================================

function G = buildCentralSquareCellVerts(P, o)
% STANDARD SQUARE CELL with side = P.L
Lside = P.L;  Wside = P.L;

V_base = vbox(o, Lside, Wside, P.H);
o_Ti   = o + [0 0 P.H];
V_Ti   = vbox(o_Ti, Lside, Wside, P.H_Ti_bottom);
o_Au   = o + [0 0 (P.H + P.H_Ti_bottom)];
V_Au   = vbox(o_Au, Lside, Wside, P.H_gold_bottom);

z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;
V_left = []; V_right = [];
V_vac  = vbox([o(1) o(2) z0_top], Lside, Wside, P.H_vac);

capSide = min(P.CapSide_center, Lside);
x0_cap = o(1) + (Lside - capSide)/2;
y0_cap = o(2) + (Wside - capSide)/2;
z0_cap = z0_top + P.H_vac;
V_cap  = vbox([x0_cap, y0_cap, z0_cap], capSide, capSide, P.H_Au_top);

o_film_xL = [x0_cap - P.T_film,      y0_cap,                z0_cap];
o_film_xR = [x0_cap + capSide,       y0_cap,                z0_cap];
V_film_xL = vbox(o_film_xL, P.T_film,               capSide, P.H_Au_top);
V_film_xR = vbox(o_film_xR, P.T_film,               capSide, P.H_Au_top);

o_film_yF = [x0_cap - P.T_film,      y0_cap - P.T_film,     z0_cap];
o_film_yB = [x0_cap - P.T_film,      y0_cap + capSide,      z0_cap];
V_film_yF = vbox(o_film_yF, capSide + 2*P.T_film, P.T_film,   P.H_Au_top);
V_film_yB = vbox(o_film_yB, capSide + 2*P.T_film, P.T_film,   P.H_Au_top);

x_in_min = x0_cap - P.T_film;
x_in_max = x0_cap + capSide + P.T_film;
y_in_min = y0_cap - P.T_film;
y_in_max = y0_cap + capSide + P.T_film;

x_base_min = o(1); x_base_max = o(1) + Lside;
y_base_min = o(2); y_base_max = o(2) + Wside;

T_left_frame  = max(0, x_in_min - x_base_min);
T_right_frame = max(0, x_base_max - x_in_max);
T_front_frame = max(0, y_in_min - y_base_min);
T_back_frame  = max(0,  y_base_max - y_in_max);

z0_frame = z0_cap; H_frame = P.H_Au_top;

V_frame_xL = []; V_frame_xR = []; V_frame_yF = []; V_frame_yB = [];
if T_left_frame  > 0, V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame,  Wside, H_frame); end
if T_right_frame > 0, V_frame_xR = vbox([x_in_max,   y_base_min, z0_frame], T_right_frame, Wside, H_frame); end
if T_front_frame > 0, V_frame_yF = vbox([x_in_min,   y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame); end
if T_back_frame  > 0, V_frame_yB = vbox([x_in_min,   y_in_max,   z0_frame], max(0,x_in_max-x_in_min), T_back_frame,  H_frame); end

G.base  = V_base;  G.ti   = V_Ti;    G.au   = V_Au;
G.left  = V_left;  G.vac  = V_vac;   G.right= V_right;
G.cap   = V_cap;   G.fxL  = V_film_xL; G.fxR = V_film_xR;
G.fyF   = V_film_yF; G.fyB= V_film_yB;
G.frmL  = V_frame_xL; G.frmR = V_frame_xR;
G.frmF  = V_frame_yF; G.frmB = V_frame_yB;
end

function G = buildCellVerts(P, o, isCut)
V_base = vbox(o, P.L, P.W, P.H);
o_Ti = o + [0 0 P.H];
V_Ti = vbox(o_Ti, P.L, P.W, P.H_Ti_bottom);
o_Au = o + [0 0 (P.H + P.H_Ti_bottom)];
V_Au = vbox(o_Au, P.L, P.W, P.H_gold_bottom);

z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;
L_vac_eff = min(max(P.L_vac,0), P.L);
L_left  = max(0, (P.L - L_vac_eff)/2);
L_right = max(0, P.L - (L_left + L_vac_eff));

o_left  = [o(1),                 o(2), z0_top];
o_vac   = [o(1) + L_left,        o(2), z0_top];
o_right = [o(1) + L_left + L_vac_eff, o(2), z0_top];

V_left  = vbox(o_left,  L_left,    P.W, P.H_vac);
V_vac   = vbox(o_vac,   L_vac_eff, P.W, P.H_vac);
V_right = vbox(o_right, L_right,   P.W, P.H_vac);

x0_cap = o(1) + (P.L - P.L_Au_top)/2;
y0_cap = o(2) + (P.W - P.W_Au_top)/2;
z0_cap = z0_top + P.H_vac;
o_cap  = [x0_cap, y0_cap, z0_cap];
V_cap  = vbox(o_cap, P.L_Au_top, P.W_Au_top, P.H_Au_top);

o_film_xL = [x0_cap - P.T_film,      y0_cap,              z0_cap];
o_film_xR = [x0_cap + P.L_Au_top,    y0_cap,              z0_cap];
V_film_xL = vbox(o_film_xL, P.T_film,             P.W_Au_top, P.H_Au_top);
V_film_xR = vbox(o_film_xR, P.T_film,             P.W_Au_top, P.H_Au_top);
o_film_yF = [x0_cap - P.T_film,      y0_cap - P.T_film,   z0_cap];
o_film_yB = [x0_cap - P.T_film,      y0_cap + P.W_Au_top, z0_cap];
V_film_yF = vbox(o_film_yF, P.L_Au_top + 2*P.T_film, P.T_film,   P.H_Au_top);
V_film_yB = vbox(o_film_yB, P.L_Au_top + 2*P.T_film, P.T_film,   P.H_Au_top);

x_in_min = x0_cap - P.T_film;
x_in_max = x0_cap + P.L_Au_top + P.T_film;
y_in_min = y0_cap - P.T_film;
y_in_max = y0_cap + P.W_Au_top + P.T_film;

x_base_min = o(1); x_base_max = o(1) + P.L;
y_base_min = o(2); y_base_max = o(2) + P.W;

T_left_frame  = max(0, x_in_min - x_base_min);
T_right_frame = max(0, x_base_max - x_in_max);
T_front_frame = max(0, y_in_min - y_base_min);
T_back_frame  = max(0,  y_base_max - y_in_max);

z0_frame = z0_cap; H_frame = P.H_Au_top;

V_frame_xL = []; V_frame_xR = []; V_frame_yF = []; V_frame_yB = [];
if T_left_frame  > 0, V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame,  P.W, H_frame); end
if T_right_frame > 0, V_frame_xR = vbox([x_in_max,   y_base_min, z0_frame], T_right_frame, P.W, H_frame); end
if T_front_frame > 0, V_frame_yF = vbox([x_in_min,   y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame); end
if T_back_frame  > 0, V_frame_yB = vbox([x_in_min,   y_in_max,   z0_frame], max(0,x_in_max-x_in_min), T_back_frame,  H_frame); end

if isCut
    W_pack_ext = P.W_Au_top + 2*P.T_film;
    if P.W > W_pack_ext
        yMinCut = o(2);
        yMaxCut = min(y0_cap + P.W_Au_top + P.T_film, o(2) + P.W);
    else
        yMinCut = o(2); yMaxCut = o(2) + P.W;
    end
    V_base    = cropYBox(V_base,    yMinCut,yMaxCut);
    V_Ti      = cropYBox(V_Ti,      yMinCut,yMaxCut);
    V_Au      = cropYBox(V_Au,      yMinCut,yMaxCut);
    V_left    = cropYBox(V_left,    yMinCut,yMaxCut);
    V_vac     = cropYBox(V_vac,     yMinCut,yMaxCut);
    V_right   = cropYBox(V_right,   yMinCut,yMaxCut);
    V_cap     = cropYBox(V_cap,     yMinCut,yMaxCut);
    V_film_xL = cropYBox(V_film_xL, yMinCut,yMaxCut);
    V_film_xR = cropYBox(V_film_xR, yMinCut,yMaxCut);
    V_film_yF = cropYBox(V_film_yF, yMinCut,yMaxCut);
    V_film_yB = cropYBox(V_film_yB, yMinCut,yMaxCut);

    y_base_min = yMinCut; y_base_max = yMaxCut;
    T_left_frame  = max(0, x_in_min - x_base_min);
    T_right_frame = max(0, x_base_max - x_in_max);
    T_front_frame = max(0, y_in_min - y_base_min);
    T_back_frame  = max(0,  y_base_max - y_in_max);

    V_frame_xL = []; V_frame_xR = []; V_frame_yF = []; V_frame_yB = [];
    if T_left_frame  > 0, V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame,  yMaxCut-yMinCut, H_frame); end
    if T_right_frame > 0, V_frame_xR = vbox([x_in_max,   y_base_min, z0_frame], T_right_frame, yMaxCut-yMinCut, H_frame); end
    if T_front_frame > 0, V_frame_yF = vbox([x_in_min,   y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame); end
    if T_back_frame  > 0, V_frame_yB = vbox([x_in_min,   y_in_max,   z0_frame], max(0,x_in_max-x_in_min), T_back_frame,  H_frame); end
end

G.base  = V_base;  G.ti   = V_Ti;    G.au   = V_Au;
G.left  = V_left;  G.vac  = V_vac;   G.right= V_right;
G.cap   = V_cap;   G.fxL  = V_film_xL; G.fxR = V_film_xR;
G.fyF   = V_film_yF; G.fyB= V_film_yB;
G.frmL  = V_frame_xL; G.frmR = V_frame_xR;
G.frmF  = V_frame_yF; G.frmB = V_frame_yB;
end

function Gt = rotZ_around(G, Rz, Cxy)
fn = fieldnames(G);
for i=1:numel(fn)
    V = G.(fn{i});
    if isempty(V), Gt.(fn{i}) = []; continue; end
    Vc = V;
    Vc(:,1) = V(:,1) - Cxy(1);
    Vc(:,2) = V(:,2) - Cxy(2);
    Vrot = (Rz * Vc.').';
    Vrot(:,1) = Vrot(:,1) + Cxy(1);
    Vrot(:,2) = Vrot(:,2) + Cxy(2);
    Gt.(fn{i}) = Vrot;
end
end

function drawGeom(ax,G, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, addLegend)
drawPatch(ax,G.base, c_base, edgeCol,lw, addLegend,'SiO2', 1, F);
drawPatch(ax,G.ti,   c_ti,   edgeCol,lw, addLegend,'Ti',   1, F);
drawPatch(ax,G.au,   c_au,   edgeCol,lw, addLegend,'Au',   1, F);
drawPatch(ax,G.left, c_hfo2, edgeCol,lw, addLegend,'HfO2', 1, F);
drawPatch(ax,G.vac,  c_vac,  edgeCol,lw, addLegend,'vacuum', a_vac, F);
drawPatch(ax,G.right,c_hfo2, edgeCol,lw, false,'', 1, F, true);
drawPatch(ax,G.cap,  c_au,   edgeCol,lw, false,'', 1, F, true);
drawPatch(ax,G.fxL,  c_ti,   edgeCol,lw, false,'', 1, F, true);
drawPatch(ax,G.fxR,  c_ti,   edgeCol,lw, false,'', 1, F, true);
drawPatch(ax,G.fyF,  c_ti,   edgeCol,lw, false,'', 1, F, true);
drawPatch(ax,G.fyB,  c_ti,   edgeCol,lw, false,'', 1, F, true);
drawPatch(ax,G.frmL, c_base, edgeCol,lw, false,'', 1, F, true);
drawPatch(ax,G.frmR, c_base, edgeCol,lw, false,'', 1, F, true);
drawPatch(ax,G.frmF, c_base, edgeCol,lw, false,'', 1, F, true);
drawPatch(ax,G.frmB, c_base, edgeCol,lw, false,'', 1, F, true);
end

function FL = buildFillerLayers(o, Lx, Wy, hvec)
z = o(3);
FL.V1 = vbox([o(1) o(2) z], Lx, Wy, hvec(1)); z = z + hvec(1);
FL.V2 = vbox([o(1) o(2) z], Lx, Wy, hvec(2)); z = z + hvec(2);
FL.V3 = vbox([o(1) o(2) z], Lx, Wy, hvec(3)); z = z + hvec(3);
FL.V4 = vbox([o(1) o(2) z], Lx, Wy, hvec(4));
end

function drawFillerLayers(ax, FL, c1, c2, c3, c4, edgeCol, lw, F, legendName, offLegend)
if nargin<10, legendName=''; end
if nargin<11, offLegend=false; end

if ~isempty(legendName) && ~offLegend
    patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1,'EdgeColor',edgeCol,'LineWidth',lw, ...
        'DisplayName',legendName);
else
    patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1,'EdgeColor',edgeCol,'LineWidth',lw, ...
        'HandleVisibility','off');
end
patch(ax,'Vertices',FL.V2,'Faces',F,'FaceColor',c2,'EdgeColor',edgeCol,'LineWidth',lw, ...
    'HandleVisibility','off');
patch(ax,'Vertices',FL.V3,'Faces',F,'FaceColor',c3,'EdgeColor',edgeCol,'LineWidth',lw, ...
    'HandleVisibility','off');
patch(ax,'Vertices',FL.V4,'Faces',F,'FaceColor',c4,'EdgeColor',edgeCol,'LineWidth',lw, ...
    'HandleVisibility','off');
end

function V = vbox(o,Lx,Wy,Hz)
V = [ o;
      o + [Lx 0 0];
      o + [0 Wy 0];
      o + [0 0 Hz];
      o + [Lx Wy 0];
      o + [Lx 0 Hz];
      o + [0 Wy Hz];
      o + [Lx Wy Hz] ];
end

function Vc = cropYBox(V, yMin, yMax)
if isempty(V), Vc = []; return; end
oB  = V(1,:); LxB = V(2,1)-V(1,1); WyB = V(3,2)-V(1,2); HzB = V(4,3)-V(1,3);
y1 = oB(2); y2 = oB(2)+WyB;
if y2 <= yMin || y1 >= yMax, Vc = []; return; end
ya = max(y1, yMin); yb = min(y2, yMax); Wc = yb - ya;
if Wc <= 0, Vc = []; else, Vc = vbox([oB(1) ya oB(3)], LxB, Wc, HzB); end
end

function F = faces6()
F = [1 2 5 3; 3 5 8 7; 1 3 7 4; 2 6 8 5; 1 4 6 2; 4 7 8 6];
end

function ph = drawPatch(axh,V,col,edgeCol,lw, addToLegend, name, faceAlpha, F, offLegend)
if nargin<10, offLegend=false; end
if isempty(V), ph = []; return; end
if addToLegend && ~offLegend && ~isempty(name)
    ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
               'EdgeColor',edgeCol,'LineWidth',lw,'DisplayName',name);
else
    ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
               'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
end
end

function pitch = calcPitchCutY(P)
W_pack_ext = P.W_Au_top + 2*P.T_film;
if P.W > W_pack_ext
    pitch = min(P.W, 0.5*(P.W + P.W_Au_top + 2*P.T_film));
else
    pitch = P.W;
end
end
